#!/usr/bin/env python3
"""
parameter_sweep_low_gauge.py

Perform a sweep over pivot parameters ``b`` and ``k`` for multiple gauge
groups (U1, SU2, SU3) in the low‑flip regime.  This script expects the
base configuration ``config.yaml`` to be set up for a 4×4 lattice and
``results_dir`` to point to ``results_sweep_low_gauge``.  For each
combination of ``b`` and ``k`` the script writes a temporary config
file, runs the simulation pipeline and extracts the string tension
``σ`` by fitting the log of the Wilson loop averages against the loop
area.  The aggregated results are written to
``results_sweep_low_gauge/sweep_sigma_low_gauge.csv``.

To use:

    chmod +x scripts/parameter_sweep_low_gauge.py
    python scripts/parameter_sweep_low_gauge.py

"""

import os
import subprocess
import yaml
import numpy as np
import pandas as pd


def main():
    # Load the base configuration
    with open('config.yaml') as f:
        base = yaml.safe_load(f)

    # Ensure results directory exists
    results_dir = 'results_sweep_low_gauge'
    os.makedirs(results_dir, exist_ok=True)

    records = []
    b_values = base.get('pivot', {}).get('b_values', [])
    k_values = base.get('pivot', {}).get('logistic_k_values', [])
    gauge_groups = [g.upper() for g in base.get('gauge_groups', [])]

    for b in b_values:
        for k in k_values:
            # Prepare configuration for this sweep point
            cfg = dict(base)
            cfg['pivot'] = dict(base['pivot'])
            cfg['pivot']['b'] = b
            cfg['pivot']['logistic_k'] = k
            cfg_file = f'cfg_low_gauge_b{b}_k{k}.yaml'
            with open(cfg_file, 'w') as out:
                yaml.safe_dump(cfg, out)

            # Run the simulation pipeline with the temporary config
            subprocess.run(['python', 'run_simulation.py', '--config', cfg_file], check=True)

            # For each gauge group, compute sigma from its Wilson loop data
            for G in gauge_groups:
                csv_path = os.path.join(results_dir, f'wilson_{G}.csv')
                if not os.path.exists(csv_path):
                    # Skip if the gauge group was not produced
                    continue
                df = pd.read_csv(csv_path)
                sizes = df['size'].astype(float).values
                area = sizes ** 2
                values = df['real'].astype(float) + 1j * df['imag'].astype(float)
                logav = np.log(np.abs(values))
                sigma, _ = np.polyfit(area, logav, 1)
                records.append({'gauge_group': G, 'b': b, 'k': k, 'sigma': float(sigma)})

    # Save aggregated results
    out_csv = os.path.join(results_dir, 'sweep_sigma_low_gauge.csv')
    pd.DataFrame(records).to_csv(out_csv, index=False)
    print(f'✅ {out_csv} generated')


if __name__ == '__main__':
    main()